<?php
/**
 * Plugin Name: TopStrona Safe
 * Author: TopStrona Wojciech Wróblewski
 * Author URI: https://topstrona.pl
 * Description: Zaawansowana ochrona WordPress
 * Version: 1.3.2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* =========================
   KONFIGURACJA
========================= */

define( 'TS_SAFE_OPTION', 'topstrona_safe_enabled' );
define( 'TS_LOGO_URL', 'https://topstrona.pl/wp-content/uploads/2025/12/horizontal_on_transparent_2000x899px_by_logaster.png' );

define( 'TS_HT_START', '# TopStrona Safe – START' );
define( 'TS_HT_END',   '# TopStrona Safe – END' );

/* =========================
   BLOKADA EDYCJI PLIKÓW (POPRAWNA)
========================= */

add_filter( 'file_mod_allowed', function( $allowed ) {
    if ( get_option( TS_SAFE_OPTION, '0' ) === '1' ) {
        return false; // blokada edytora motywów i wtyczek
    }
    return $allowed;
}, 99 );

/* =========================
   HTACCESS
========================= */

function ts_safe_htaccess_rules() {
return <<<HT
# --------------------------------------------------------------
# TopStrona Safe – START
# --------------------------------------------------------------

Options -Indexes

<FilesMatch "^\.">
    Require all denied
</FilesMatch>

<FilesMatch "\.(log|sql|bak|old|save|swp|swx)$">
    Require all denied
</FilesMatch>

<Files wp-config.php>
    Require all denied
</Files>

<FilesMatch "^(readme\.html|license\.txt)$">
    Require all denied
</FilesMatch>

<FilesMatch "wp-content/uploads/.*\.php$">
    Require all denied
</FilesMatch>

<Directory "wp-content">
    <FilesMatch "\.php$">
        Require all denied
    </FilesMatch>
</Directory>

<Files xmlrpc.php>
    Require all denied
</Files>

<IfModule mod_rewrite.c>
RewriteEngine On
RewriteRule ^wp-admin/includes/ - [F,L]
RewriteRule !^wp-includes/ - [S=3]
RewriteRule ^wp-includes/[^/]+\.php$ - [F,L]
RewriteRule ^wp-includes/js/tinymce/langs/.+\.php$ - [F,L]
RewriteRule ^wp-includes/theme-compat/ - [F,L]
</IfModule>

<IfModule mod_rewrite.c>
RewriteEngine On
RewriteCond %{HTTP_USER_AGENT} SemrushBot [NC,OR]
RewriteCond %{HTTP_USER_AGENT} SemrushBot-SA [NC,OR]
RewriteCond %{HTTP_USER_AGENT} SemrushBot-BA [NC,OR]
RewriteCond %{HTTP_USER_AGENT} SemrushBot-SI [NC]
RewriteRule .* - [F,L]
RewriteCond %{HTTP_USER_AGENT} AhrefsBot [NC]
RewriteRule .* - [F,L]
RewriteCond %{HTTP_USER_AGENT} bingbot [NC]
RewriteRule .* - [F,L]
</IfModule>

# --------------------------------------------------------------
# TopStrona Safe – END
# --------------------------------------------------------------
HT;
}

function ts_safe_add_htaccess() {
    $f = ABSPATH . '.htaccess';
    if ( ! file_exists( $f ) || ! is_writable( $f ) ) return;
    if ( strpos( file_get_contents( $f ), TS_HT_START ) !== false ) return;
    file_put_contents( $f, PHP_EOL . ts_safe_htaccess_rules() . PHP_EOL, FILE_APPEND );
}

function ts_safe_remove_htaccess() {
    $f = ABSPATH . '.htaccess';
    if ( ! file_exists( $f ) || ! is_writable( $f ) ) return;
    $c = file_get_contents( $f );
    $p = '/' . preg_quote( TS_HT_START, '/' ) . '.*?' . preg_quote( TS_HT_END, '/' ) . '/s';
    file_put_contents( $f, trim( preg_replace( $p, '', $c ) ) . PHP_EOL );
}

/* =========================
   AKCJE
========================= */

register_activation_hook( __FILE__, function() {
    update_option( TS_SAFE_OPTION, '1' );
    ts_safe_add_htaccess();
});

add_action( 'update_option_' . TS_SAFE_OPTION, function( $old, $new ) {
    $new === '1' ? ts_safe_add_htaccess() : ts_safe_remove_htaccess();
}, 10, 2 );

function ts_safe_enabled() {
    return get_option( TS_SAFE_OPTION, '0' ) === '1';
}

/* =========================
   MENU ADMINA (IKONA + POZYCJA)
========================= */

add_action( 'admin_menu', function() {
    add_menu_page(
        'TopStrona Safe',
        'TopStrona Safe',
        'manage_options',
        'topstrona-safe',
        'ts_safe_admin',
        'dashicons-shield',
        2 // wysoko w menu
    );
});

/* =========================
   PANEL ADMINA
========================= */

function ts_safe_admin() {
    $on = ts_safe_enabled();
?>
<div class="wrap ts-safe-wrap">

<?php if ( $on ) : ?>
<div class="ts-safe-active-box">
    <img src="<?php echo esc_url( TS_LOGO_URL ); ?>">
    <div class="ts-safe-active-text">Ochrona aktywna</div>
</div>
<?php endif; ?>

<form method="post" action="options.php" class="ts-safe-form">
<?php settings_fields( 'ts_safe' ); ?>
<input type="hidden" name="<?php echo TS_SAFE_OPTION; ?>" value="0">

<label class="ts-switch">
    <input type="checkbox" name="<?php echo TS_SAFE_OPTION; ?>" value="1" <?php checked( $on ); ?> onchange="this.form.submit();">
    <span class="ts-slider"></span>
</label>

<div class="ts-switch-label">
    <?php echo $on ? 'Ochrona WŁĄCZONA' : 'Ochrona WYŁĄCZONA'; ?>
</div>
</form>

<h2>Status systemu</h2>
<ul class="ts-status">
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Blokada XML-RPC</li>
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Ochrona przed botami</li>
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Blokada PHP w uploadach</li>
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Ukrycie plików systemowych</li>
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Ograniczenie REST API</li>
<li class="<?php echo $on ? 'ok' : 'off'; ?>">✔ Blokada edycji plików</li>
</ul>
</div>

<style>
.ts-safe-wrap{max-width:820px}
.ts-safe-active-box{background:#ecfdf5;border:2px solid #22c55e;border-radius:16px;padding:30px;text-align:center;margin-bottom:30px}
.ts-safe-active-box img{max-width:320px;margin-bottom:15px}
.ts-safe-active-text{font-size:22px;font-weight:700;color:#15803d}
.ts-safe-form{display:flex;align-items:center;gap:20px;margin-bottom:30px}
.ts-switch{position:relative;width:64px;height:34px}
.ts-switch input{display:none}
.ts-slider{position:absolute;inset:0;background:#e5e7eb;border-radius:34px;transition:.3s}
.ts-slider:before{content:"";position:absolute;width:26px;height:26px;left:4px;bottom:4px;background:#fff;border-radius:50%;transition:.3s}
.ts-switch input:checked+.ts-slider{background:#22c55e}
.ts-switch input:checked+.ts-slider:before{transform:translateX(30px)}
.ts-switch-label{font-size:18px;font-weight:600}
.ts-status li.ok{color:#15803d;font-weight:600}
.ts-status li.off{color:#b91c1c;font-weight:600}
</style>
<?php
}

/* =========================
   OPCJA
========================= */

add_action( 'admin_init', function() {
    register_setting( 'ts_safe', TS_SAFE_OPTION );
});
